﻿using HostMgd.ApplicationServices;
using HostMgd.EditorInput;
using System.Numerics;
using System.Runtime.Intrinsics;
using Teigha.DatabaseServices;
using Teigha.Geometry;
using Teigha.GraphicsInterface;
using Teigha.Runtime;
using Platform = HostMgd;


[assembly: CommandClass(typeof(TestCurve.AllCommands))]

namespace TestCurve
{
    public class AllCommands
    {
        static Document Ndoc = Application.DocumentManager.MdiActiveDocument;
        static Database db = Ndoc.Database;
        static Editor editor = Application.DocumentManager.MdiActiveDocument.Editor;

        [CommandMethod("TEST", CommandFlags.UsePickSet)]
        public void Example()
        {
            PromptPointResult pPtRes;
            PromptPointOptions pPtOpts = new PromptPointOptions("")
            {
                Message = "\nУкажите точку для расчета: "
            };
            pPtRes = editor.GetPoint(pPtOpts);
            if (pPtRes.Status == PromptStatus.OK)// если точка получена, получаем криволинейный объект на котором расположена точка по ID
            {
                Point3d StartPoint = pPtRes.Value;
                Transaction tr = db.TransactionManager.StartTransaction();
                ObjectId? curveId = SelectFilteredObjectsInWindow(10, 0.01, StartPoint);
                if (curveId != null)
                {
                    // принимаем объект как кривую и находим касательную
                    //получаем угол вектора по отношению к оси Х
                    Entity? lineObj = curveId.Value.GetObject(OpenMode.ForRead) as Entity;
                    if (lineObj != null)
                    {
                        Vector3d tempVector = (lineObj as Curve).GetFirstDerivative(StartPoint);
                        double rangle = tempVector.GetAngleTo(Vector3d.XAxis);
                        double angle = rangle * 180 / Math.PI;
                        editor.WriteMessage($"\nПроизводная = {tempVector}.");
                        editor.WriteMessage($"\nУгол = {angle}.");
                        VisualizeVector(StartPoint, tempVector, angle);
                    }
                    DrawPoint(StartPoint);
                }
                tr.Commit();
            }
        }

        private ObjectId? SelectFilteredObjectsInWindow(int iterCount, double delta, Point3d p)
        {
            ObjectId? result = null;
            TypedValue[] values = new TypedValue[8];
            values.SetValue(new TypedValue((int)DxfCode.Operator, "<or"), 0);
            values.SetValue(new TypedValue((int)DxfCode.Start, "ARC"), 1);
            values.SetValue(new TypedValue((int)DxfCode.Start, "CIRCLE"), 2);
            values.SetValue(new TypedValue((int)DxfCode.Start, "LWPOLYLINE"), 3);
            values.SetValue(new TypedValue((int)DxfCode.Start, "HELIX"), 4);
            values.SetValue(new TypedValue((int)DxfCode.Start, "ELLIPSE"), 5);
            values.SetValue(new TypedValue((int)DxfCode.Start, "SPLINE"), 6);
            values.SetValue(new TypedValue((int)DxfCode.Operator, "or>"), 7);
            SelectionFilter filter = new SelectionFilter(values);
            for (int i = 1; i <= iterCount; i++)
            {
                delta = delta * i * i;
                Point3d p1 = new Point3d(p.X - delta, p.Y - delta, 0.0);
                Point3d p2 = new Point3d(p.X + delta, p.Y + delta, 0.0);
                PromptSelectionResult res = editor.SelectCrossingWindow(p1, p2, filter);
                SelectionSet ss = res.Value;
                if (ss.Count != 0)
                {
                    ObjectId[] oid = ss.GetObjectIds();
                    if (oid.Length >= 1)
                    {
                        result = oid[0];
                        editor.WriteMessage($"\nКривая: {result.Value.ObjectClass.Name}.");
                        break;
                    }
                }
                else
                {
                    continue;
                }
            }
            return result;
        }

        private void DrawPoint(Point3d pt)
        {
            using (Transaction tr = db.TransactionManager.StartTransaction())
            {
                // Открываем таблицу блоков для записи
                BlockTable bt = (BlockTable)tr.GetObject(
                    db.BlockTableId, OpenMode.ForRead);
                BlockTableRecord btr = (BlockTableRecord)tr.GetObject(
                    bt[BlockTableRecord.ModelSpace], OpenMode.ForWrite);

                // Создаем точку с координатами (10, 20, 0)
                DBPoint point = new DBPoint(pt);
                point.ColorIndex = 1;
                // Добавляем точку в модель
                btr.AppendEntity(point);
                tr.AddNewlyCreatedDBObject(point, true);

                tr.Commit();
            }
        }

        private static void VisualizeVector(Point3d startPoint, Vector3d vector, double angle)
        {
            using (Transaction tr = db.TransactionManager.StartTransaction())
            {
                BlockTable bt = (BlockTable)tr.GetObject(
                    db.BlockTableId, OpenMode.ForRead);
                BlockTableRecord btr = (BlockTableRecord)tr.GetObject(
                    bt[BlockTableRecord.ModelSpace], OpenMode.ForWrite);

                // 1. Рисуем основную линию вектора
                Point3d endPoint = startPoint + vector;
                Line vectorLine = new Line(startPoint, endPoint);
                vectorLine.ColorIndex = 1; // Красный

                btr.AppendEntity(vectorLine);
                tr.AddNewlyCreatedDBObject(vectorLine, true);

                // 2. Рисуем стрелку на конце вектора
                DrawArrow(startPoint, endPoint, btr, tr);

                // 3. Добавляем текст с информацией о векторе
                DrawVectorInfo(startPoint, vector, angle, btr, tr);

                tr.Commit();
            }
        }
        private static void DrawArrow(Point3d start, Point3d end, BlockTableRecord btr, Transaction tr)
        {
            if (start.DistanceTo(end) < 0.001) return;

            double arrowSize = start.DistanceTo(end) * 0.1; // 10% от длины вектора

            // Направление вектора
            Vector3d direction = end - start;
            direction = direction.GetNormal();

            // Перпендикулярные векторы для стрелки
            Vector3d perp1 = direction.RotateBy(Math.PI / 6, Vector3d.ZAxis);
            Vector3d perp2 = direction.RotateBy(-Math.PI / 6, Vector3d.ZAxis);

            // Точки стрелки
            Point3d arrowPoint1 = end - perp1 * arrowSize;
            Point3d arrowPoint2 = end - perp2 * arrowSize;

            // Рисуем линии стрелки
            Line arrowLine1 = new Line(end, arrowPoint1);
            Line arrowLine2 = new Line(end, arrowPoint2);
            arrowLine1.ColorIndex = 1;
            arrowLine2.ColorIndex = 1;

            btr.AppendEntity(arrowLine1);
            btr.AppendEntity(arrowLine2);
            tr.AddNewlyCreatedDBObject(arrowLine1, true);
            tr.AddNewlyCreatedDBObject(arrowLine2, true);
        }

        private static void DrawVectorInfo(Point3d start, Vector3d vector, double angle, BlockTableRecord btr, Transaction tr)
        {
            // Текст с информацией о векторе
            DBText infoText = new DBText();
            infoText.Position = start + new Vector3d(0, -5, 0);
            infoText.Height = 2.5;
            infoText.TextString = $"Вектор: ({vector.X:F2}, {vector.Y:F2}, {vector.Z:F2}) Длина: {vector.Length:F2} Угол: {angle:F2}";
            infoText.ColorIndex = 3; // Зеленый

            btr.AppendEntity(infoText);
            tr.AddNewlyCreatedDBObject(infoText, true);
        }
    }
}
